package main

import (
	"fmt"
	"image"
	"image/color"
	"image/draw"
	_ "image/jpeg"
	"image/png"
	"io/ioutil"
	"log"
	"os"

	"github.com/golang/freetype/truetype"
	"github.com/rwcarlsen/goexif/exif"
	_ "github.com/rwcarlsen/goexif/tiff"
	"golang.org/x/image/font"
	"golang.org/x/image/math/fixed"
)

func checkError(err error) {
	if err != nil {
		log.Fatal(err)
	}
}

func imageToRGBA(src image.Image) *image.RGBA {

	// No conversion needed if image is an *image.RGBA.
	if dst, ok := src.(*image.RGBA); ok {
		return dst
	}

	// Use the image/draw package to convert to *image.RGBA.
	b := src.Bounds()
	dst := image.NewRGBA(image.Rect(0, 0, b.Dx(), b.Dy()))
	draw.Draw(dst, dst.Bounds(), src, b.Min, draw.Src)
	return dst
}

func addLabel(img *image.RGBA, ttffont *truetype.Font, fontSize, x, y, offset, lineHeight int, label string) {
	col := color.RGBA{200, 100, 0, 255}
	point := fixed.Point26_6{fixed.I(x), fixed.I(img.Bounds().Dy() - (y + lineHeight*offset))}

	d := &font.Drawer{
		Dst: img,
		Src: image.NewUniform(col),
		Face: truetype.NewFace(ttffont, &truetype.Options{
			Size: float64(fontSize),
			DPI:  72,
		}),
		Dot: point,
	}
	d.DrawString(label)
}

func TrimString(s string) string {
	s = s[:len(s)-1][1:]

	return s
}

func main() {
	// User variables
	inputDirectory := "../"
	outputDirectory := "../output/"
	watermark := "PtGrph  OnTake"
	fontFile := "SpaceMono-Bold.ttf"
	textPadding := []int{32, 32}
	lineHeight := 102
	fontSize := 96

	// Code
	files, err := ioutil.ReadDir(inputDirectory)
	checkError(err)

	ttfData, err := ioutil.ReadFile(fontFile)
	checkError(err)
	ttffont, err := truetype.Parse(ttfData)
	checkError(err)

	for _, file := range files {
		if !file.IsDir() {
			inputFile, err := os.Open(inputDirectory + file.Name())
			checkError(err)
			defer inputFile.Close()

			x, err := exif.Decode(inputFile)
			checkError(err)
			imageDateTime, err := x.DateTime()
			checkError(err)
			cameraModel, err := x.Get("Model")
			checkError(err)
			cameraMake, err := x.Get("Make")
			checkError(err)
			imageFNumber, err := x.Get("FNumber")
			checkError(err)
			imageFocalLengthIn35mmFilm, err := x.Get("FocalLengthIn35mmFilm")
			checkError(err)
			imageISO, err := x.Get("ISOSpeedRatings")
			checkError(err)
			fmt.Println("--------" + file.Name() + "--------")
			fmt.Println(imageDateTime)
			fmt.Println(cameraModel)
			fmt.Println(cameraMake)
			fmt.Println(imageFNumber)
			fmt.Println(imageFocalLengthIn35mmFilm)
			fmt.Println(imageISO)
			inputFile2, err := os.Open(inputDirectory + file.Name())
			checkError(err)
			defer inputFile2.Close()
			imageData, _, err := image.Decode(inputFile2)
			checkError(err)
			imageDataRGBA := imageToRGBA(imageData)
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 0, lineHeight, fmt.Sprintf("ƒ'   %smm", imageFocalLengthIn35mmFilm))
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 1, lineHeight, fmt.Sprintf("ƒ    %s", TrimString(fmt.Sprintf("%s", imageFNumber))))
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 2, lineHeight, fmt.Sprintf("ISO  %s", imageISO))
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 3, lineHeight, TrimString(fmt.Sprintf("%s", cameraMake))+" "+TrimString(fmt.Sprintf("%s", cameraModel)))
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 4, lineHeight, fmt.Sprintf("%s", imageDateTime))
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 5, lineHeight, file.Name())
			addLabel(imageDataRGBA, ttffont, fontSize, textPadding[0], textPadding[1], 6, lineHeight, watermark)
			outputFile, err := os.Create(outputDirectory + file.Name())
			checkError(err)
			defer outputFile.Close()
			fmt.Println("Exporting...")
			err = png.Encode(outputFile, imageDataRGBA)
			checkError(err)
		}
	}

}
